// SPDX-License-Identifier: BSD-3-Clause
// The license is BSD because decode_dec is inspired by
// ultoa_invert in avrlibc.
// Copyright (c) 2005,2007  Dmitry Xmelkov
// Copyright (c) 2018-2019 Rupert Eibauer

/* Rup's very own printk */

/* 
 * I was never happy with the original printf from avr-libc.
 * It was not only a stack hog, it was too big, slow,
 * required lots of libs.
 * printf-min was too minimalistic for me.
 * 
 * This implementation can be considered its own subsystem.
 * During compilation, a script collects all format strings and generates
 * a feature list of format strings we neeed to support.
 * This implementation shows undefined behaviour on invalid format strings,
 * but that's not a problem because they are checked for correctness at compile time.
 * Anyway, format strings are only allowed in progmem, not in normal memory.
 * 
 */

#include <stdarg.h>

#ifndef __TEST__

#include <linux/types.h>
#include <linux/cdev.h>
#include <rupsched/vdevprintf.h>

#include <linux/compiler.h>
#include <linux/string.h>

#endif

#include "printf_config.h"

extern char* __ultoa_invert(u32 num, char const *buf, u8 base);

/* FIXME: */
#define CONFIG_PRINTF_SPC
#define CONFIG_PRINTF_PLUS

#if !defined(CONFIG_PRINTF_d) && !defined(CONFIG_PRINTF_u)
#undef CONFIG_PRINTF_SIGN
#else
#if defined(CONFIG_PRINTF_PLUS) || defined(CONFIG_PRINTF_SPC)
#define CONFIG_PRINTF_SIGN
#endif
#endif

#ifdef CONFIG_PRINTF_ld
#define CONFIG_PRINTF_d
#define CONFIG_PRINTF_l
#endif
#ifdef CONFIG_PRINTF_lo
#define CONFIG_PRINTF_o
#define CONFIG_PRINTF_l
#endif
#ifdef CONFIG_PRINTF_lu
#define CONFIG_PRINTF_u
#define CONFIG_PRINTF_l
#endif
#ifdef CONFIG_PRINTF_lx
#define CONFIG_PRINTF_x
#define CONFIG_PRINTF_l
#endif

#if defined(CONFIG_PRINTF_ZPAD) && !defined(CONFIG_PRINTF_SPCPAD)
#define DEFAULT_PAD '0'
#else
#define DEFAULT_PAD ' '
#endif

#ifndef CONFIG_PRINTF_ZPAD
#define num_is_pad(ch, width)  0
#elif CONFIG_PRINTF_MAX_WIDTH < 10
#define num_is_pad(ch, width)  (ch == '0')
#else
#define num_is_pad(ch, width)  (ch == '0' && !width)
#endif

#if defined(CONFIG_PRINTF_NUM_LPAD) || defined(CONFIG_PRINTF_STR_LPAD)
#define CONFIG_PRINTF_LPAD
#endif

#if defined(CONFIG_PRINTF_NUMPAD) || defined(CONFIG_PRINTF_STRPAD)
#define CONFIG_PRINTF_WIDTH
#endif


#if ((('d' | 'u') & 0x08) == 0) && ((('x' | 'o') & 0x08) == 0x08)
#define ch_is_d_or_u(ch)  ((ch & 0x08) == 0)
#else
#define ch_is_d_or_u(ch)  (ch == 'd' || ch == 'u')
#endif

#if (('x' &  0x04) == 0x04) && ((('d' | 'o'| 'u') & 0x04) == 0)
#define ch_is_ix(ch) (ch & 0x04)
#else
#define ch_is_x(ch) (ch == 'x')
#endif

#if  ((('d' & 'o') & 0x20) == 0) && ((('u' | 'x') & 0x20) == 0x20)
#define ch_is_d_or_o(ch)    ((ch & 1) == 0)
#define ch_is_d_know_no_o   ch_is_d_or_o
#else
#define ch_is_d_or_o(ch)    ('ch' == 'd' || 'ch' == 'o')
#endif


#if !defined(CONFIG_PRINTF_o) && defined(ch_is_d_know_no_o)
// #define ch_is_d   ch_is_d_know_no_o
#define ch_is_d(ch)  (ch == 'd')
#else
#define ch_is_d(ch)  (ch == 'd')
#endif

#define FLAG_LONG     0x10
#define FLAG_LPAD     0x20
#define FLAG_STRLPAD  0x20
#define FLAG_NUMLPAD  0x20
#define DO_STRPAD     1
#define DO_NUMPAD     1


#ifndef CONFIG_PRINTF_STRPAD
#undef DO_STRPAD
#undef FLAG_STRLPAD
#define DO_STRPAD 0
#define FLAG_STRLPAD 0
#endif
#ifndef CONFIG_PRINTF_STR_LPAD
#undef FLAG_STRLPAD
#define FLAG_STRLPAD 0
#endif

#ifndef CONFIG_PRINTF_NUMPAD
#undef DO_NUMPAD
#undef FLAG_NUMPAD
#define DO_NUMPAD   0
#define FLAG_NUMPAD 0
#endif
#ifndef CONFIG_PRINTF_NUM_LPAD
#undef FLAG_NUMLPAD
#define FLAG_NUMLPAD 0
#endif

#ifdef __TEST__
#define printf_long_t int
#define printf_int_t short
#define printf_ull_t unsigned long
#else
#define printf_int_t int
#define printf_long_t long
#define printf_ull_t unsigned long long
#endif

#ifdef CONFIG_PRINTF_l

#define printf_arg_t printf_long_t

#else

#define printf_arg_t printf_int_t

#undef  FLAG_LONG
#define FLAG_LONG 0

#endif


#if defined(CONFIG_PRINTF_ld) || defined(CONFIG_PRINTF_lu)

#define printf_dec_t unsigned printf_long_t
#define printf_dec2_t printf_ull_t
#define DEC_CONST 0x33333333UL
#define DEC_SHIFT 32

#else /* CONFIG_PRINTF_ld || CONFIG_PRINTF_lu */

#define printf_dec_t  unsigned printf_int_t
#define printf_dec2_t unsigned printf_long_t
#define DEC_CONST 0x3333U
#define DEC_SHIFT 16

#endif /* CONFIG_PRINTF_ld || CONFIG_PRINTF_lu */

static __maybe_unused
printf_dec_t div10(printf_dec_t value) {
	printf_dec2_t long_value;

	long_value = DEC_CONST;
#if DEC_SHIFT == 32
	/* This seems to be a GCC bug!*/
	long_value <<= 16;
	long_value |= DEC_CONST;
#endif
	long_value *= (value >> 1) + 1;

	return long_value >> DEC_SHIFT;
}

static __maybe_unused
char *decode_dec(char * buf, printf_dec_t value) {
	u8 old_lo;
	u8 new_lo;
	while(value) {
		old_lo = value;
		value = div10(value);
		new_lo = value;
		new_lo *= 10;
		*buf++ = old_lo - new_lo + '0';
	}
	return buf;
}

static __maybe_unused
char transform_hex(char ch) {
	if (ch > 9) {
		ch += 'a' - 10 - '0';
	}
	ch += '0';
	return ch;
}

#ifdef CONFIG_PRINTF_lx
#define printf_hex_t unsigned printf_long_t
#else
#define printf_hex_t unsigned printf_int_t
#endif

static __maybe_unused
char *decode_hex(char * buf, printf_hex_t value) {
	while (value) {
		*buf++ = transform_hex(value & 0x0f);
		value &=  ~(printf_hex_t)0x0f;
		if (value == (printf_hex_t)0)
			break;
		*buf++ = transform_hex((u8)value >> 4);
		value >>= 8;
	}
	return buf;
}

#ifdef CONFIG_PRINTF_lo
#define printf_oct_t unsigned printf_long_t
#else
#define printf_oct_t unsigned printf_int_t
#endif

static __maybe_unused
char *decode_oct(char * buf, printf_oct_t value) {
	while(value) {
		*buf++ = (value & 7) + '0';
		value >>= 3;
	}
	return buf;
}


#define fmt_getc(fmt)    (*fmt++)

#ifdef CONFIG_PRINTF_o
#define BUF_SIZE_o  (sizeof(printf_oct_t) * 268 + 99) / 100
#else
#define BUF_SIZE_o  0
#endif

#if defined(CONFIG_PRINTF_d) || defined(CONFIG_PRINTF_u)
#define BUF_SIZE_d  (sizeof(printf_dec_t) * 241 + 99) / 100
#else
#define BUF_SIZE_d  0
#endif

#ifdef CONFIG_PRINTF_x
#define BUF_SIZE_x sizeof(printf_hex_t) * 2
#else
#define BUF_SIZE_x  0
#endif

#define MAX(x, y)  ((x) > (y)) ? (x) : (y)
#define MAX3(x, y, z)  MAX(x, MAX(y, z))
#define BUF_SIZE MAX3(BUF_SIZE_o, BUF_SIZE_d, BUF_SIZE_x)

static __maybe_unused
char *decode_num(char *buf, printf_arg_t arg, char ch) {

#if defined(CONFIG_PRINTF_d) || defined(CONFIG_PRINTF_u)
#if defined(CONFIG_PRINTF_x) || defined(CONFIG_PRINTF_o)
	if (ch_is_d_or_u(ch))
#endif
		return decode_dec(buf, arg);
#endif
#if defined(CONFIG_PRINTF_x) && defined(CONFIG_PRINTF_o)
	if (ch_is_x(ch))
		return decode_hex(buf, arg);
	return decode_oct(buf, arg);
#elif defined(CONFIG_PRINTF_x)
	return decode_hex(buf, arg);
#else
	return decode_oct(buf, arg);
#endif
}

void vdevprintf(char __rodata *fmt, struct ocdev *dev, va_list args) {
	char ch;
	char width;
	char sign;
	char pad;
	char flags;
	char buf[BUF_SIZE] __maybe_unused;

	while (1) {
		ch = fmt_getc(fmt);
		if (!ch)
			break;
#ifdef CONFIG_PRINTF_NONE
		putc(dev, ch);
		continue;
#endif /* CONFIG_PRINTF_NONE */
		if (ch != '%') {
			putc(dev, ch);
			continue;
		}
		width = 0;
		sign = 0;
		pad = DEFAULT_PAD;
		flags = 0;
nextfmt: __maybe_unused;
		ch = fmt_getc(fmt);
#ifdef CONFIG_PRINTF_WIDTH
		if (ch >= '0' && ch <= '9') {
			if (!num_is_pad(ch, width)) {
#if CONFIG_PRINTF_MAX_WIDTH > 9
				width *= 10;
				width += ch - '0';
#else
				width = ch - '0';
#endif
			} else {
				pad = ch;
			}
			goto nextfmt;
		}
#endif /* CONFIG_PRINTF_WIDTH */

#ifdef CONFIG_PRINTF_HASHFUNC
#define hashfunc(ch) ((ch ^ 8) + (9 ^ (ch + 12)))
#define casehash(ch) case (((hashfunc(ch) >> 4) ^ hashfunc(ch)) & 0xf)
		u8 hash = hashfunc(ch);
		hash = ((hash >> 4) ^ hash) & 0xf;
#define switchhash(ch) switch(hash)
#else
#define casehash(x)    case x
#define switchhash(ch) switch(ch)
#endif
		
		switchhash (ch) {
			casehash(0):                       return;
#ifdef CONFIG_PRINTF_PERCENT
			casehash('%'): putc(dev, ch);      break;
#endif
#ifdef CONFIG_PRINTF_l
			casehash('l'): flags |= FLAG_LONG; goto nextfmt;
#endif
#ifdef CONFIG_PRINTF_LPAD
			casehash('-'): flags |= FLAG_LPAD; goto nextfmt;
#endif
#ifdef CONFIG_PRINTF_SIGN
#ifdef CONFIG_PRINTF_SPC
			casehash(' '):
#endif
#ifdef CONFIG_PRINTF_PLUS
			casehash('+'):
#endif
				       sign = ch;          goto nextfmt;
#endif /* CONFIG_PRINTF_SIGN */
#ifdef CONFIG_PRINTF_c
			casehash('c'):
				ch = va_arg(args, int);
				putc(dev, ch);
				break;
#endif
#ifdef CONFIG_PRINTF_s
			casehash('s'):
#endif
#ifdef CONFIG_PRINTF_S
			casehash('S'):
#endif
#if defined(CONFIG_PRINTF_s) || defined(CONFIG_PRINTF_S)

#define ch_is_lower(ch)    (ch & ('a' ^ 'A'))
#ifndef CONFIG_PRINTF_s
#define ch_s_is_lower(ch)    0
#elif !defined(CONFIG_PRINTF_S)
#define ch_s_is_lower(ch)    1
#else
#define ch_s_is_lower(ch)  ch_is_lower(ch)
#endif

			{
				char *src = va_arg(args, char*);
				char c;
#ifndef CONFIG_PRINTF_STRPAD
				width = 0;
#endif
				if (width) {
					width -= strlen(src);
				}

				if (!(flags & FLAG_STRLPAD)) while (width > 0) {
					width--;
					putc(dev, ' ');
				}
				while ((c = ch_s_is_lower(ch) ?
					(*src++) :
					*((char __rodata*)(uintptr_t)(src++))
				)) {
					putc(dev, c);
				}

				while (width > 0) {
					width--;
					putc(dev, ' ');
				}
				break;
			}
#endif

#ifdef CONFIG_PRINTF_d
			casehash('d'):
#endif
#ifdef CONFIG_PRINTF_u
			casehash('u'):
#endif
#ifdef CONFIG_PRINTF_x
			casehash('x'):
#endif
#ifdef CONFIG_PRINTF_o
			casehash('o'):
#endif
#if defined(CONFIG_PRINTF_d) || defined(CONFIG_PRINTF_u) || defined(CONFIG_PRINTF_x) || defined(CONFIG_PRINTF_o)
			{
				printf_arg_t arg;
				char *bufp;

				if (flags & FLAG_LONG) {
					arg = va_arg(args, long);
				} else {
					arg = va_arg(args, int);
				}

				if (ch_is_d(ch)) {
					if (arg < 0) {
						sign = '-';
						arg = -arg;
					}
				}
				if (!(flags & FLAG_LONG)) {
					arg = (unsigned int)arg;
				}

				bufp = decode_num(buf, arg, ch);
				if (bufp == buf) // How to get rid of that compiler warning without increasing code size?
					*bufp++ = '0'; // Print at least one zero

				width -= bufp - buf;
#ifndef CONFIG_PRINTF_NUMPAD
				width = 0;
				pad = '0';
#endif

				if (sign && pad == '0') {
					width--;
					putc(dev, sign);
				}

				if (!(flags & FLAG_NUMLPAD)) while (width > 0) {
					width--;
					putc(dev, pad);
				}

				if (sign && pad == ' ') {
					putc(dev, sign);
				}

				while (bufp != buf) {
					putc(dev, *--bufp);
				}
				while (width > 0) {
					width--;
					putc(dev, ' ');
				}
			}
			break;
#endif /* Numeric support */
		}
	}
}
