// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2018-2019 Rupert Eibauer. All rights reserved.

#include <linux/kernel.h>
#include <linux/printk.h>
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/sched.h>

#include <asm/system.h>
#include <asm/limits.h>
#include <asm/io.h>

static const int panic_timeout = 20;

#define DUMP_WIDTH 0x20
#define DUMP_MASK (DUMP_WIDTH-1)

#define GRUOP_WIDTH 4
#define GROUP_MASK  (GRUOP_WIDTH - 1)

void dump_memory(void) {
	u16 p = 0;
	for (p = RAM_START; p <= RAM_END; p++) {
		if ((p & DUMP_MASK) == 0x0)
			printk(KERN_EMERG "%03x ", p);
		printk(KERN_EMERG "%02x", *(u8*)p);
		if ((p & GROUP_MASK) == GROUP_MASK)
			printk(KERN_EMERG " ");
		if ((p & DUMP_MASK) == DUMP_MASK) {
			u16 p2 = p - DUMP_MASK;
			for (p2 = p - DUMP_MASK; p2 <= p; p2++) {
				u8 ch = *(u8*)p2;
				if (ch < 32 || ch >= 127)
					ch = '.';
				printk(KERN_EMERG "%c", ch);
			}
			printk(KERN_EMERG "\n");
		}
	}
	printk(KERN_EMERG "\n");
}

void _panic(void) {
	void *sp;
	irqflags_t flags;
	void *retaddr;

	local_irq_save(flags);

	sp = (void*)in_16(SP);
	retaddr = __builtin_return_address(0);
	out_16(SP, RAM_END - 0x10); // Make sure we have enough stack

#ifdef CONFIG_SCHEDULER
	printk(KERN_EMERG "Kernel Panic: @ PC=%04x SP=%04x SREG=%02x current=%04x\n",
	((u16)retaddr - 1) << 1, sp, flags, current);
#else
	printk(KERN_EMERG "Kernel Panic: @ PC=%04x SP=%04x SREG=%02x\n",
	((u16)retaddr - 1) << 1, sp, flags);
#endif
	dump_memory();

	if (panic_timeout > 0) {
		int i;
		void (*restart)(void) __noreturn = NULL;
		printk(KERN_EMERG "Rebooting in %d seconds..", panic_timeout);
		for (i = 0; i < panic_timeout; i++) {
			mdelay(1000);
		}
		restart();
	} else {
		printk(KERN_EMERG "System halted.\n");
		for(;;)
			;
	}
}
