// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2018-2019 Rupert Eibauer. All rights reserved.

#include <linux/init.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/printk.h>

#include <asm/rodata.h>
#include <asm/irqvec.h>

struct vector_table_entry {
	void (*func)(u8 vec);
};

void bad_interrupt(u8 vec) {
	printk(KERN_EMERG "spurious interrupt %d\n", vec);
#ifdef CONFIG_PANIC
	panic();
#endif
	for (;;) ;
}

#define VECTOR(num) void _vector_ ## num(u8) __attribute__((weak, alias("bad_interrupt")));
#include <asm/vectors.h>
#undef VECTOR

#ifdef CONFIG_IRQ_COMMON_STATIC
#define vectable_attr __rodata
#else
#define vectable_attr
#endif

#define VECTABLE_ENTRY(name) { .func = name }

static struct vector_table_entry vectable_attr vector_table[NR_IRQS] = {
#define VECTOR(num) VECTABLE_ENTRY(_vector_ ## num),
#include <asm/vectors.h>
#undef VECTOR
};

void do_IRQ(u8 vector) {
	void (*func)(u8 vec);

	/* VECTORS start counting at 1. We don't want to waste the first entry in vector_table.  */
	func = vector_table[vector-1].func;
	func(vector);
	resched_after_irq();
}

#ifdef CONFIG_IRQ_COMMON_DYNAMIC
void request_irq(u8 vector, void(*func)(u8 vec)) {
	vector_table[vector - 1].func = func;
}

#if (0)
void free_irq(u8 vector) {
	vector_table[vector - 1].func = bad_interrupt;
}
#endif
#endif
