// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2008, 2014, 2018-2019 Rupert Eibauer. All rights reserved.

#include <linux/types.h>
#include <linux/timer.h>
#include <linux/jiffies.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/interrupt.h>

#include <asm/system.h>
#include <asm/io.h>

#define TIMER_RANGE	256
#define TIMER_VECTOR	IRQ_TIMER0_OVF

#ifdef TCCR0

#define TIMER_REG	TCCR0
#define TIMER_ENABLE	TIMSK
#define TIMER_ENABLE_BITS TIMSK_TOIE0
#define PRESCALE1        TCCR0_CS0_PS_1
#define PRESCALE8        TCCR0_CS0_PS_8
#define PRESCALE64       TCCR0_CS0_PS_64
#define PRESCALE256      TCCR0_CS0_PS_256
#define PRESCALE1024     TCCR0_CS0_PS_1024
#ifdef TCCR0_CS0_PS_32
#define PRESCALE32       TCCR0_CS0_PS_32
#endif
#ifdef TCCR0_CS0_PS_128
#define PRESCALE128      TCCR0_CS0_PS_128
#endif

#elif defined(TCCR0B)

#define TIMER_REG	TCCR0B
#define TIMER_ENABLE	TIMSK0
#define TIMER_ENABLE_BITS TIMSK0_TOIE0
#define PRESCALE1        TCCR0B_CS0_PS_1
#define PRESCALE8        TCCR0B_CS0_PS_8
#define PRESCALE64       TCCR0B_CS0_PS_64
#define PRESCALE256      TCCR0B_CS0_PS_256
#define PRESCALE1024     TCCR0B_CS0_PS_1024
#ifdef TCCR0B_CS0_PS_32
#define PRESCALE32       TCCR0B_CS0_PS_32
#endif
#ifdef TCCR0B_CS0_PS_128
#define PRESCALE128      TCCR0B_CS0_PS_128
#endif

#else

#error "unsupported device"

#endif


#ifdef CONFIG_DYNAMIC_TICK

#ifdef CONFIG_PRESCALE1024
#define PRESCALE PRESCALE1024
#define PRESCALE_FACTOR 1024
#define TIMER_RANGE_BUFFER 0
#elif defined(CONFIG_PRESCALE256)
#define PRESCALE PRESCALE256
#define PRESCALE_FACTOR 256
#define TIMER_RANGE_BUFFER 1
#elif defined(CONFIG_PRESCALE64)
#define PRESCALE PRESCALE64
#define PRESCALE_FACTOR 64
#define TIMER_RANGE_BUFFER 4
#elif defined(CONFIG_PRESCALE8)
#define PRESCALE PRESCALE8
#define PRESCALE_FACTOR 8
#define TIMER_RANGE_BUFFER 32
#else
#error "No prescaler defined"
#endif

#else /* CONFIG_DYNAMIC_TICK */

#define CPUCLOCKS_PER_JIFFY (CONFIG_CPUFREQ / CONFIG_HZ)

#define MINIMUM_PRESCALER  ((CPUCLOCKS_PER_JIFFY+TIMER_RANGE-1)/TIMER_RANGE)

#if CPUCLOCKS_PER_JIFFY < 128
#warning System tick maybe too fast.
#endif

#if MINIMUM_PRESCALER > 1024
#error "Minimum Prescaler too big. Increase CONFIG_HZ!"
#elif MINIMUM_PRESCALER > 256 && defined(PRESCALE1024)
#define PRESCALE PRESCALE1024
#define PRESCALE_FACTOR 1024
#elif MINIMUM_PRESCALER > 64 && defined(PRESCALE256)
#define PRESCALE PRESCALE256
#define PRESCALE_FACTOR 256
#elif MINIMUM_PRESCALER > 128 && defined(PRESCALE128)
#define PRESCALE PRESCALE128
#define PRESCALE_FACTOR 128
#elif MINIMUM_PRESCALER > 8 && defined(PRESCALE64)
#define PRESCALE PRESCALE64
#define PRESCALE_FACTOR 64
#warning prescale 64
#elif MINIMUM_PRESCALER > 8 && defined(PRESCALE32)
#define PRESCALE PRESCALE32
#define PRESCALE_FACTOR 32
#elif MINIMUM_PRESCALER > 1 && defined(PRESCALE8)
#define PRESCALE PRESCALE8
#define PRESCALE_FACTOR 8
#elif MINIMUM_PRESCALER == 1 && defined(PRESCALE1)
#define PRESCALE PRESCALE1
#define PRESCALE_FACTOR 1
#else
#error "Minimum Prescaler too small. Decrease CONFIG_HZ!"
#endif

#if PRESCALE_FACTOR == 1
#define CLOCKS_PER_JIFFY (CONFIG_CPUFREQ/CONFIG_HZ-1)
#else
#define CLOCKS_PER_JIFFY (CONFIG_CPUFREQ/CONFIG_HZ/PRESCALE_FACTOR)
#endif

#if CLOCKS_PER_JIFFY > TIMER_RANGE
#error "Fix calculation of MINIMUM_PRESCALER"
#endif

#endif /* CONFIG_DYNAMIC_TICK */


#ifdef CONFIG_DYNAMIC_TICK
volatile jiffies_t last_jiffies;
#else
volatile jiffies_t jiffies;
#endif

extern struct timer *next_timer;

static inline void sync_timer(void) {
	/* Wait until the counter changes!
	   This is a trick to make sure we don't loose a single (prescaler) tick */
	/* This makes best sense with a prescaler of 8,
	*  and limited sense with 64:
	*  with prescaler=1, we know that it will change at each
	*  tick. We do increase CLOCKS_PER_JIFFY by one to 
	*  compensate for this.
	*  With prescalers >64, we wait longer than it actually takes to
	*  serve an interrupt.
	*  It also becomes less probable to hit it during an update, if the
	*  prescaler is larger.
	*/
#if 0
	char old_tcnt;
	old_tcnt = in(TCNT0);
	while (old_tcnt == in(TCNT0))
		;
#endif
}


#ifdef CONFIG_DYNAMIC_TICK
/* This is called each time somebody wants to know the time. 
   It is not really too slow.
*/
jiffies_t get_jiffies() {
	u8 low_jiffies;

	low_jiffies = in(TCNT0);
	return last_jiffies + low_jiffies;
}

static void __update_timer(void) {
	jiffies_t next_expiry;
	s16 jiffies_diff;

	if (!next_timer)
		return;

	/* We assume that it is in the future. This can be a problem if timers
	   are too close together. FIXME!*/
	next_expiry = next_timer->expires - last_jiffies;

	if (next_expiry >= TIMER_RANGE - 2) {
		/* We don't touch it. Reduce the chance of mis-counting */
		return;
		/* Or we put it to the minimum (0) */
		jiffies_diff = -in(TCNT0) + 2;
	} else {
		/* We program the timer for the exact time it should expire */
		jiffies_diff = TIMER_RANGE - next_expiry - in(TCNT0);
	}
#if (PRESCALE_FACTOR >= 8) && (PRESCALE_FACTOR <= 64)
	sync_timer();
#endif
	out(TCNT0, in(TCNT0) + (u8)jiffies_diff);
	last_jiffies -= jiffies_diff;
}

void _update_timer(void) {
	__update_timer();
}

jiffies_t reset_timer(void) {
	jiffies_t cached_jiffies;

	cached_jiffies = last_jiffies;
	cached_jiffies += TIMER_RANGE;
	last_jiffies = cached_jiffies;
	return cached_jiffies;
}

#else /* !CONFIG_DYNAMIC_TICK */

#define __update_timer() do{}while(0)
static inline jiffies_t reset_timer(void) {
	jiffies_t cached_jiffies;

#if CLOCKS_PER_JIFFY != TIMER_RANGE

#if (PRESCALE_FACTOR >= 8) && (PRESCALE_FACTOR <= 64)
	sync_timer();
#endif
	out(TCNT0, in(TCNT0) - (u8)CLOCKS_PER_JIFFY);
#else
#warning Omitting TCNT0 update
#endif
	cached_jiffies = jiffies;
	cached_jiffies++;
	jiffies = cached_jiffies;
	return cached_jiffies;
}

#endif /* !CONFIG_DYNAMIC_TICK */

#if defined(CONFIG_DYNAMIC_TICK) && (PRESCALE_FACTOR <= 64)
#define SHORT_TIMER_COMPENSATION
#warning SHORT_TIMER_COMPENSATION
#endif

static void timer_irq(u8 irq) {
	jiffies_t cached_jiffies;

	cached_jiffies = reset_timer();

	while (next_timer && time_before_eq(next_timer->expires, cached_jiffies)) {
		void (*timer_function)(void);
#ifdef SHORT_TIMER_COMPENSATION
restart:
#endif /* SHORT_TIMER_COMPENSATION */
		timer_function = next_timer->function;
		_del_first_timer();
		timer_function();
#ifdef CONFIG_DYNAMIC_TICK
		cached_jiffies = last_jiffies + in(TCNT0);
#endif
	}
#ifdef SHORT_TIMER_COMPENSATION
	cached_jiffies = last_jiffies + in(TCNT0);
	if (next_timer && time_before_eq(next_timer->expires, cached_jiffies + (64 / PRESCALE_FACTOR))) {
		while (!time_before_eq(next_timer->expires, cached_jiffies)) {
			cached_jiffies = last_jiffies + in(TCNT0);
		}
		goto restart;
	}
#endif /* SHORT_TIMER_COMPENSATION */
	/* For dynamic ticks, we need to call it unconditionally,
	   for normal ones, update_timer is empty anyway. */
	__update_timer();
}

REQUEST_IRQ(TIMER_VECTOR, timer_irq);

static int init_timers(void) {
	out(TIMER_REG, PRESCALE);
	out(TIMER_ENABLE, in(TIMER_ENABLE) | TIMER_ENABLE_BITS);

	return 0;
}
early_initcall(init_timers);
