// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2018-2019 Rupert Eibauer. All rights reserved.

#include <linux/init.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/printk.h>

#include <asm/flash.h>
#include <asm/irqvec.h>

struct vector_table_entry {
	void (*func)(u8 vec);
};

#ifdef CONFIG_IRQ_COMMON
__attribute__((noreturn)) void bad_interrupt(u8 vec) {
	printk(KERN_EMERG "spurious interrupt %d\n", vec);
	panic();
	for (;;) ;
}
#endif

#ifdef CONFIG_IRQ_COMMON_STATIC
#define VECTABLE_ENTRY(name) { .func = name }

#define VECTOR(num) void _vector_ ## num(u8) __attribute__((weak, alias("bad_interrupt")));
#include <asm/vectors.h>
#undef VECTOR

static const struct vector_table_entry __flashdata_decl vector_table[NR_IRQS] __attribute__((used)) = {
#define VECTOR(num) VECTABLE_ENTRY(_vector_ ## num),
#include <asm/vectors.h>
#undef VECTOR
};

#else

struct vector_table_entry vector_table[NR_IRQS];

#endif

int i = 0;
void do_IRQ(u8 vector) {
	void (*func)(u8 vec);

	/* VECTORS start counting at 1. We don't want to waste the first entry in vector_table.  */
#ifdef CONFIG_IRQ_COMMON_STATIC
	func = flash_read(vector_table[vector-1].func);
#else
	func = vector_table[vector-1].func;
#endif
	func(vector);
	resched_after_irq();
}

#ifdef CONFIG_IRQ_COMMON_DYNAMIC
void request_irq(u8 vector, void(*func)(u8 vec)) {
	vector_table[vector - 1].func = func;
}
#endif

static int init_interrupts(void) {
#ifdef CONFIG_IRQ_COMMON_DYNAMIC
	int i;
	for (i = 0; i < NR_IRQS; i++) {
		vector_table[i].func = bad_interrupt;
	}
#endif
	return 0;
}

early_initcall(init_interrupts);
